<?php

namespace App\Http\Controllers\Admin;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Models\NotificationLog;
use App\Models\NotificationTemplate;
use App\Models\PaidLog;
use App\Models\Staff;
use App\Rules\FileTypeValidate;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Illuminate\Validation\Rules\Password;

class ManageStaffsController extends Controller {
    public function allStaffs() {
        $pageTitle = 'All staffs';
        $staffs    = $this->staffData();
        return view('admin.staffs.list', compact('pageTitle', 'staffs'), $this->getCommonData());
    }

    public function activeStaffs() {
        $pageTitle = 'Active staffs';
        $staffs    = $this->staffData('active');
        return view('admin.staffs.list', compact('pageTitle', 'staffs'), $this->getCommonData());
    }

    public function bannedStaffs() {
        $pageTitle = 'Banned staffs';
        $staffs    = $this->staffData('banned');
        return view('admin.staffs.list', compact('pageTitle', 'staffs'), $this->getCommonData());
    }

    public function emailUnverifiedStaffs() {
        $pageTitle = 'Email Unverified staffs';
        $staffs    = $this->staffData('emailUnverified');
        return view('admin.staffs.list', compact('pageTitle', 'staffs'), $this->getCommonData());
    }

    public function emailVerifiedStaffs() {
        $pageTitle = 'Email Verified staffs';
        $staffs    = $this->staffData('emailVerified');
        return view('admin.staffs.list', compact('pageTitle', 'staffs'), $this->getCommonData());
    }

    public function mobileUnverifiedStaffs() {
        $pageTitle = 'Mobile Unverified staffs';
        $staffs    = $this->staffData('mobileUnverified');
        return view('admin.staffs.list', compact('pageTitle', 'staffs'), $this->getCommonData());
    }

    public function mobileVerifiedStaffs() {
        $pageTitle = 'Mobile Verified staffs';
        $staffs    = $this->staffData('mobileVerified');
        return view('admin.staffs.list', compact('pageTitle', 'staffs'), $this->getCommonData());
    }

    private function getCommonData() {
        $info       = json_decode(json_encode(getIpInfo()), true);
        $mobileCode = isset($info['code']) ? implode(',', $info['code']) : '';
        $countries  = json_decode(file_get_contents(resource_path('views/partials/country.json')));
        return ['mobileCode' => $mobileCode, 'countries' => $countries];
    }

    protected function staffData($scope = null) {
        if ($scope) {
            $staffs = Staff::$scope();
        } else {
            $staffs = Staff::query();
        }

        return $staffs->searchable(['username', 'email'])->orderBy('id', 'desc')->paginate(getPaginate());
    }

    public function detail($id) {
        $staff     = Staff::findOrFail($id);
        $pageTitle = 'Staff Detail - ' . $staff->username;

        $widget['today_loan_collection_count']     = PaidLog::where('staff_id', $id)->whereDate('created_at', now()->format('Y-m-d'))->where('loan_id', '!=', 0)->count();
        $widget['today_loan_collection_amount']    = PaidLog::where('staff_id', $id)->whereDate('created_at', now()->format('Y-m-d'))->where('loan_id', '!=', 0)->sum('amount');
        $widget['today_savings_collection_count']  = PaidLog::where('staff_id', $id)->whereDate('created_at', now()->format('Y-m-d'))->where('savings_id', '!=', 0)->count();
        $widget['today_savings_collection_amount'] = PaidLog::where('staff_id', $id)->whereDate('created_at', now()->format('Y-m-d'))->where('savings_id', '!=', 0)->sum('amount');

        $countries = json_decode(file_get_contents(resource_path('views/partials/country.json')));
        return view('admin.staffs.detail', compact('pageTitle', 'staff', 'widget', 'countries'));
    }

    public function login($id) {
        $staff = Staff::findOrFail($id);
        Auth::guard('staff')->login($staff);
        return to_route('staff.home');
    }

    public function status(Request $request, $id) {
        $user = Staff::findOrFail($id);
        if ($user->status == Status::USER_ACTIVE) {
            $request->validate([
                'reason' => 'required|string|max:255',
            ]);
            $user->status     = Status::USER_BAN;
            $user->ban_reason = $request->reason;
            $notify[]         = ['success', 'Staff banned successfully'];
        } else {
            $user->status     = Status::USER_ACTIVE;
            $user->ban_reason = null;
            $notify[]         = ['success', 'Staff unbanned successfully'];
        }
        $user->save();
        return back()->withNotify($notify);
    }

    public function showNotificationSingleForm($id) {
        $user = Staff::findOrFail($id);
        if (!gs('en') && !gs('sn') && !gs('pn')) {
            $notify[] = ['warning', 'Notification options are disabled currently'];
            return to_route('admin.staffs.detail', $user->id)->withNotify($notify);
        }
        $pageTitle = 'Send Notification to ' . $user->username;
        return view('admin.staffs.notification_single', compact('pageTitle', 'user'));
    }

    public function sendNotificationSingle(Request $request, $id) {
        $request->validate([
            'message' => 'required',
            'via'     => 'required|in:email,sms,push',
            'subject' => 'required_if:via,email,push',
            'image'   => ['nullable', 'image', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
        ]);

        if (!gs('en') && !gs('sn') && !gs('pn')) {
            $notify[] = ['warning', 'Notification options are disabled currently'];
            return to_route('admin.dashboard')->withNotify($notify);
        }

        $imageUrl = null;
        if ($request->via == 'push' && $request->hasFile('image')) {
            $imageUrl = fileUploader($request->image, getFilePath('push'));
        }

        $template = NotificationTemplate::where('act', 'DEFAULT')->where($request->via . '_status', Status::ENABLE)->exists();
        if (!$template) {
            $notify[] = ['warning', 'Default notification template is not enabled'];
            return back()->withNotify($notify);
        }

        $user = Staff::findOrFail($id);
        notify($user, 'DEFAULT', [
            'subject' => $request->subject,
            'message' => $request->message,
        ], [$request->via], pushImage: $imageUrl);
        $notify[] = ['success', 'Notification sent successfully'];
        return back()->withNotify($notify);
    }

    public function showNotificationAllForm() {
        if (!gs('en') && !gs('sn') && !gs('pn')) {
            $notify[] = ['warning', 'Notification options are disabled currently'];
            return to_route('admin.dashboard')->withNotify($notify);
        }

        $notifyToStaff = Staff::notifyToStaff();
        $staffs        = Staff::active()->count();
        $pageTitle     = 'Notification to Verified staffs';

        if (session()->has('SEND_NOTIFICATION') && !request()->email_sent) {
            session()->forget('SEND_NOTIFICATION');
        }

        return view('admin.staffs.notification_all', compact('pageTitle', 'staffs', 'notifyToStaff'));
    }

    public function sendNotificationAll(Request $request) {
        $request->validate([
            'via'                          => 'required|in:email,sms,push',
            'message'                      => 'required',
            'subject'                      => 'required_if:via,email,push',
            'start'                        => 'required|integer|gte:1',
            'batch'                        => 'required|integer|gte:1',
            'being_sent_to'                => 'required',
            'cooling_time'                 => 'required|integer|gte:1',
            'number_of_top_deposited_user' => 'required_if:being_sent_to,topDepositedStaffs|integer|gte:0',
            'number_of_days'               => 'required_if:being_sent_to,notLoginStaffs|integer|gte:0',
            'image'                        => ["nullable", 'image', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
        ], [
            'number_of_days.required_if'               => "Number of days field is required",
            'number_of_top_deposited_user.required_if' => "Number of top deposited user field is required",
        ]);

        if (!gs('en') && !gs('sn') && !gs('pn')) {
            $notify[] = ['warning', 'Notification options are disabled currently'];
            return to_route('admin.dashboard')->withNotify($notify);
        }

        $template = NotificationTemplate::where('act', 'DEFAULT')->where($request->via . '_status', Status::ENABLE)->exists();
        if (!$template) {
            $notify[] = ['warning', 'Default notification template is not enabled'];
            return back()->withNotify($notify);
        }

        if ($request->being_sent_to == 'selectedStaffs') {
            if (session()->has("SEND_NOTIFICATION")) {
                $request->merge(['user' => session()->get('SEND_NOTIFICATION')['user']]);
            } else {
                if (!$request->user || !is_array($request->user) || empty($request->user)) {
                    $notify[] = ['error', "Ensure that the staff field is populated when sending an email to the designated staff group"];
                    return back()->withNotify($notify);
                }
            }
        }

        $scope     = $request->being_sent_to;
        $userQuery = Staff::oldest()->active()->$scope();

        if (session()->has("SEND_NOTIFICATION")) {
            $totalUserCount = session('SEND_NOTIFICATION')['total_user'];
        } else {
            $totalUserCount = (clone $userQuery)->count() - ($request->start - 1);
        }

        if ($totalUserCount <= 0) {
            $notify[] = ['error', "Notification recipients were not found among the selected staff base."];
            return back()->withNotify($notify);
        }

        $imageUrl = null;

        if ($request->via == 'push' && $request->hasFile('image')) {
            if (session()->has("SEND_NOTIFICATION")) {
                $request->merge(['image' => session()->get('SEND_NOTIFICATION')['image']]);
            }
            if ($request->hasFile("image")) {
                $imageUrl = fileUploader($request->image, getFilePath('push'));
            }
        }

        $staffs = (clone $userQuery)->skip($request->start - 1)->limit($request->batch)->get();

        foreach ($staffs as $user) {
            notify($user, 'DEFAULT', [
                'subject' => $request->subject,
                'message' => $request->message,
            ], [$request->via], pushImage: $imageUrl);
        }

        return $this->sessionForNotification($totalUserCount, $request);
    }

    private function sessionForNotification($totalUserCount, $request) {
        if (session()->has('SEND_NOTIFICATION')) {
            $sessionData = session("SEND_NOTIFICATION");
            $sessionData['total_sent'] += $sessionData['batch'];
        } else {
            $sessionData               = $request->except('_token');
            $sessionData['total_sent'] = $request->batch;
            $sessionData['total_user'] = $totalUserCount;
        }

        $sessionData['start'] = $sessionData['total_sent'] + 1;

        if ($sessionData['total_sent'] >= $totalUserCount) {
            session()->forget("SEND_NOTIFICATION");
            $message = ucfirst($request->via) . " notifications were sent successfully";
            $url     = route("admin.staffs.notification.all");
        } else {
            session()->put('SEND_NOTIFICATION', $sessionData);
            $message = $sessionData['total_sent'] . " " . $sessionData['via'] . "  notifications were sent successfully";
            $url     = route("admin.staffs.notification.all") . "?email_sent=yes";
        }
        $notify[] = ['success', $message];
        return redirect($url)->withNotify($notify);
    }

    public function countBySegment($methodName) {
        return Staff::active()->$methodName()->count();
    }

    public function list() {
        $query = Staff::active();
        if (request()->search) {
            $query->where(function ($q) {
                $q->where('email', 'like', '%' . request()->search . '%')->orWhere('username', 'like', '%' . request()->search . '%');
            });
        }
        $staffs = $query->orderBy('id', 'desc')->paginate(getPaginate());
        return response()->json([
            'success' => true,
            'staffs'  => $staffs,
            'more'    => $staffs->hasMorePages(),
        ]);
    }

    public function notificationLog($id) {
        $user      = Staff::findOrFail($id);
        $pageTitle = 'Notifications Sent to ' . $user->username;
        $logs      = NotificationLog::where('staff_id', $id)->with('staff')->orderBy('id', 'desc')->paginate(getPaginate());
        return view('admin.reports.notification_history', compact('pageTitle', 'logs', 'user'));
    }

    public function update(Request $request, $id) {
        $staff = Staff::findOrFail($id);

        $countryData  = json_decode(file_get_contents(resource_path('views/partials/country.json')));
        $countryArray = (array) $countryData;
        $countries    = implode(',', array_keys($countryArray));

        $countryCode = $request->country;
        $country     = $countryData->$countryCode->country;
        $dialCode    = $countryData->$countryCode->dial_code;

        $request->validate([
            'firstname' => 'required',
            'lastname'  => 'required',
            'email'     => 'required|string|email|unique:staffs,email,' . $staff->id,
            'mobile'    => ['required', 'regex:/^([0-9]*)$/', Rule::unique('staffs')->where('dial_code', $request->mobile_code)->ignore($id)],
            'captcha'   => 'sometimes|required',
            'country'   => 'required|in:' . $countries,
        ], [
            'firstname.required' => 'The first name field is required',
            'lastname.required'  => 'The last name field is required',
        ]);

        $staff->mobile    = $request->mobile;
        $staff->firstname = $request->firstname;
        $staff->lastname  = $request->lastname;
        $staff->email     = $request->email;

        $staff->address      = $request->address;
        $staff->city         = $request->city;
        $staff->state        = $request->state;
        $staff->zip          = $request->zip;
        $staff->country_name = @$country;
        $staff->dial_code    = $dialCode;
        $staff->country_code = $countryCode;

        $staff->ev     = $request->ev ? Status::VERIFIED : Status::UNVERIFIED;
        $staff->sv     = $request->sv ? Status::VERIFIED : Status::UNVERIFIED;
        $staff->ts     = $request->ts ? Status::ENABLE : Status::DISABLE;
        $staff->status = $request->status ? Status::ENABLE : Status::DISABLE;
        $staff->save();

        $notify[] = ['success', 'Staff updated successfully'];
        return back()->withNotify($notify);
    }

    public function register(Request $request) {
        $passwordValidation = Password::min(6);
        if (gs('secure_password')) {
            $passwordValidation = $passwordValidation->mixedCase()->numbers()->symbols()->uncompromised();
        }

        $countryData  = (array) json_decode(file_get_contents(resource_path('views/partials/country.json')));
        $countryCodes = implode(',', array_keys($countryData));
        $mobileCodes  = implode(',', array_column($countryData, 'dial_code'));
        $countries    = implode(',', array_column($countryData, 'country'));

        $request->validate([
            'firstname'    => 'required',
            'lastname'     => 'required',
            'email'        => 'required|string|email|unique:staffs',
            'password'     => ['required', 'confirmed', $passwordValidation],
            'mobile'       => ['required', 'regex:/^([0-9]*)$/', Rule::unique('staffs')->where('dial_code', $request->mobile_code)],
            'username'     => 'required|unique:users|min:6',
            'captcha'      => 'sometimes|required',
            'country_code' => 'required|in:' . $countryCodes,
            'country'      => 'required|in:' . $countries,
            'mobile_code'  => 'required|in:' . $mobileCodes,
        ], [
            'firstname.required' => 'The first name field is required',
            'lastname.required'  => 'The last name field is required',
        ]);

        //Staff Create
        $staff               = new Staff();
        $staff->firstname    = $request->firstname;
        $staff->lastname     = $request->lastname;
        $staff->email        = strtolower(trim($request->email));
        $staff->password     = Hash::make($request->password);
        $staff->username     = trim($request->username);
        $staff->country_code = $request->country_code;
        $staff->country_name = @$request->country;
        $staff->mobile       = $request->mobile_code . $request->mobile;
        $staff->dial_code    = $request->mobile_code;
        $staff->status       = Status::ENABLE;
        $staff->ev           = gs('ev') ? Status::NO : Status::YES;
        $staff->sv           = gs('sv') ? Status::NO : Status::YES;
        $staff->ts           = Status::DISABLE;
        $staff->tv           = Status::ENABLE;
        $staff->save();

        $notify[] = ['success', 'Staff added successfully'];
        return back()->withNotify($notify);
    }

    public function checkUser(Request $request) {
        $exist['data'] = false;
        $exist['type'] = null;
        if ($request->email) {
            $exist['data']  = Staff::where('email', $request->email)->exists();
            $exist['type']  = 'email';
            $exist['field'] = 'Email';
        }
        if ($request->mobile) {
            $exist['data']  = Staff::where('mobile', $request->mobile)->where('dial_code', $request->mobile_code)->exists();
            $exist['type']  = 'mobile';
            $exist['field'] = 'Mobile';
        }
        if ($request->username) {
            $exist['data']  = Staff::where('username', $request->username)->exists();
            $exist['type']  = 'username';
            $exist['field'] = 'Username';
        }
        return response($exist);
    }
}
