<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Lib\FormProcessor;
use App\Lib\RequiredConfig;
use App\Models\SavingsPlan;
use App\Models\TimeInterval;
use Illuminate\Http\Request;

class SavingsPlanController extends Controller {
    public function index() {
        $pageTitle   = 'Saving Plans';
        $savingPlans = SavingsPlan::searchable(['name'])->orderByDesc('id')->paginate(getPaginate());
        $days        = TimeInterval::select('name', 'day')->get()->toArray();
        return view('admin.plan.savings', compact('pageTitle', 'savingPlans', 'days'));
    }

    public function create() {
        $pageTitle = 'Add New Loan Plan';
        $days      = TimeInterval::all();
        return view('admin.plan.savings_form', compact('pageTitle', 'days'));
    }

    public function edit($id) {
        $pageTitle = 'Edit Loan Plan';
        $plan      = SavingsPlan::findOrFail($id);
        $days      = TimeInterval::all();
        $form      = $plan->form;
        return view('admin.plan.savings_form', compact('pageTitle', 'plan', 'days', 'form'));
    }

    private function validation($request, $formProcessor) {
        $validation = [
            'name'                 => 'required',
            'installment_interval' => 'required|integer|exists:time_intervals,day',
            'per_installment'      => 'required|numeric|gt:0',
            'giveable_amount'      => 'required|numeric|gt:0',
            'total_installment'    => 'required|integer|min:1',
            'description'          => 'required',
            'fixed_late_fee'       => 'required|numeric|min:0',
            'percent_late_fee'     => 'required|numeric|min:0',
            'field_name.*'         => 'sometimes|required',
        ];
        $savingsAmount = $request->per_installment * $request->total_installment;
        if ($savingsAmount > $request->giveable_amount) {
            $notify[] = ['error', 'Giveable amount must be greater than savings amount'];
            return back()->withNotify($notify);
        }

        $generatorValidation = $formProcessor->generatorValidation();
        $validation          = array_merge($validation, $generatorValidation['rules']);
        $request->validate($validation, $generatorValidation['messages']);
    }

    public function store(Request $request, $id = 0) {
        $formProcessor = new FormProcessor();
        $this->validation($request, $formProcessor);
        if ($id) {
            $savingPlan   = SavingsPlan::findOrFail($id);
            $generate     = $formProcessor->generate('savings_form', true, 'id', $savingPlan->form_id);
            $notification = "Savings plan updated successfully";
        } else {
            $generate     = $formProcessor->generate('savings_form');
            $savingPlan   = new SavingsPlan();
            $notification = "Savings plan created successfully";
        }
        $savingPlan->form_id              = @$generate->id ?? 0;
        $savingPlan->name                 = $request->name;
        $savingPlan->savings_amount       = $request->per_installment * $request->total_installment;
        $savingPlan->giveable_amount      = $request->giveable_amount;
        $savingPlan->installment          = $request->per_installment;
        $savingPlan->installment_interval = $request->installment_interval;
        $savingPlan->total_installment    = $request->total_installment;
        $savingPlan->fixed_late_fee       = $request->fixed_late_fee;
        $savingPlan->percent_late_fee     = $request->percent_late_fee;
        $savingPlan->description          = $request->description;
        $savingPlan->save();

        RequiredConfig::configured('saving_plan');
        $notify[] = ['success', $notification];
        return back()->withNotify($notify);
    }

    public function status($id) {
        return SavingsPlan::changeStatus($id);
    }
}
