<?php

namespace App\Http\Controllers\Staff;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\Intended;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;

class AuthorizationController extends Controller {
    protected function checkCodeValidity($staff, $addMin = 2) {
        if (!$staff->ver_code_send_at) {
            return false;
        }
        if ($staff->ver_code_send_at->addMinutes($addMin) < Carbon::now()) {
            return false;
        }
        return true;
    }

    public function authorizeForm() {
        $staff = auth()->guard('staff')->user();
        if (!$staff->status) {
            $pageTitle = 'Banned';
            $type      = 'ban';
        } else if (!$staff->ev) {
            $type           = 'email';
            $pageTitle      = 'Verify Email';
            $notifyTemplate = 'EVER_CODE';
        } else if (!$staff->sv) {
            $type           = 'sms';
            $pageTitle      = 'Verify Mobile Number';
            $notifyTemplate = 'SVER_CODE';
        } else if (!$staff->tv) {
            $pageTitle = '2FA Verification';
            $type      = '2fa';
        } else {
            return to_route('staff.home');
        }

        if (!$this->checkCodeValidity($staff) && ($type != '2fa') && ($type != 'ban')) {
            $staff->ver_code         = verificationCode(6);
            $staff->ver_code_send_at = Carbon::now();
            $staff->save();
            notify($staff, $notifyTemplate, [
                'code' => $staff->ver_code,
            ], [$type]);
        }

        return view('staff.auth.authorization.' . $type, compact('staff', 'pageTitle'));

    }

    public function sendVerifyCode($type) {
        $staff = auth()->guard('staff')->user();

        if ($this->checkCodeValidity($staff)) {
            $targetTime = $staff->ver_code_send_at->addMinutes(2)->timestamp;
            $delay      = $targetTime - time();
            throw ValidationException::withMessages(['resend' => 'Please try after ' . $delay . ' seconds']);
        }

        $staff->ver_code         = verificationCode(6);
        $staff->ver_code_send_at = Carbon::now();
        $staff->save();

        if ($type == 'email') {
            $type           = 'email';
            $notifyTemplate = 'EVER_CODE';
        } else {
            $type           = 'sms';
            $notifyTemplate = 'SVER_CODE';
        }

        notify($staff, $notifyTemplate, [
            'code' => $staff->ver_code,
        ], [$type]);

        $notify[] = ['success', 'Verification code sent successfully'];
        return back()->withNotify($notify);
    }

    public function emailVerification(Request $request) {
        $request->validate([
            'code' => 'required',
        ]);

        $staff = auth()->guard('staff')->user();

        if ($staff->ver_code == $request->code) {
            $staff->ev               = Status::VERIFIED;
            $staff->ver_code         = null;
            $staff->ver_code_send_at = null;
            $staff->save();

            $redirection = Intended::getRedirection();
            return $redirection ? $redirection : to_route('staff.home');
        }
        throw ValidationException::withMessages(['code' => 'Verification code didn\'t match!']);
    }

    public function mobileVerification(Request $request) {
        $request->validate([
            'code' => 'required',
        ]);

        $staff = auth()->guard('staff')->user();
        if ($staff->ver_code == $request->code) {
            $staff->sv               = Status::VERIFIED;
            $staff->ver_code         = null;
            $staff->ver_code_send_at = null;
            $staff->save();
            $redirection = Intended::getRedirection();
            return $redirection ? $redirection : to_route('staff.home');
        }
        throw ValidationException::withMessages(['code' => 'Verification code didn\'t match!']);
    }

    public function g2faVerification(Request $request) {
        $staff = auth()->guard('staff')->user();
        $request->validate([
            'code' => 'required',
        ]);
        $response = verifyG2fa($staff, $request->code);
        if ($response) {
            return to_route('staff.home');
        } else {
            $notify[] = ['error', 'Wrong verification code'];
            return back()->withNotify($notify);
        }
    }
}
